//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Properties;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
	/// <summary>
	/// Represents a constructor parameter that is created using a factory.
	/// </summary>
	[AttributeUsage(AttributeTargets.Class, AllowMultiple = true, Inherited = false)]
	public sealed class FactoryParameterAttribute : ConstructorParameterAttribute
	{
		private readonly Type factoryType;

		/// <summary>
		/// Initialize a new instance of the <see cref="FactoryParameterAttribute"/> with the parameter ordinal, the type and the factory type used to create the parameter.
		/// </summary>
		/// <param name="order">The ordinal posistion of the parameter.</param>
		/// <param name="parameterType">The type of parameter.</param>
		/// <param name="factoryType">The factory type used to create the parameter value. The factory must implement <see cref="IAnonymousObjectFactory"/>.</param>
		public FactoryParameterAttribute(int order, Type parameterType, Type factoryType)
			: base(order, parameterType)
		{
			if (factoryType == null)
			{
				throw new ArgumentNullException("factoryType");
			}
		
			this.factoryType = factoryType;
		}		

		/// <summary>
		/// Gets or sets the factory type used to create the parameter value.
		/// </summary>
		/// <value>
		/// The factory type used to create the parameter value.
		/// </value>
		public Type FactoryType
		{
			get { return factoryType; }			
		}

		/// <summary>
		/// When overridden in a class, gets the <see cref="FactoryParameterStrategy"/> used to create the parameter value.
		/// </summary>
		/// <param name="objectConfiguration">The object configuration used to retrieve the value.</param>
		/// <returns>A <see cref="FactoryParameterStrategy"/> object.</returns>
        protected internal override IObjectFactory<TObjectToCreate, TObjectConfiguration> GetParameterFactory<TObjectToCreate, TObjectConfiguration>(TObjectConfiguration objectConfiguration)
		{
            return (IObjectFactory<TObjectToCreate, TObjectConfiguration>)Activator.CreateInstance(this.factoryType);
		}	
	}
}
